package com.genexus.printing;

import java.io.*;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.awt.*;
import java.awt.event.*;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.nio.file.*;
import javax.print.*;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.printing.PDFPageable;

public class Printer
{
    private static native boolean GxPrintDoc(String fileName, String AlternateIniPath);
    private static native boolean GxIsAlive();
    private static native boolean GxShutdown();
    
    private static int BUFFER = 2048;
    private static String tempDirectory;
    private String baseUrl;
    private ArrayList<String> tempFiles;
    private boolean isReady;
    private static FileHandler loggerFile;
    private Logger log = Logger.getLogger("gxPrinter");
    private static Boolean isWindows;
    
    Printer( String baseURL) 
    {
        this.baseUrl = baseURL;
        init();
    }
    
    public static boolean isWindows()
    {
        if(isWindows == null)
        {
            isWindows = new Boolean((System.getProperty("os.name", "NONE").toLowerCase().indexOf("windows") >= 0));
        }
        return isWindows.booleanValue();
    }	
	
    private static String getTempDirectory()
    {
        if (tempDirectory == null)
        {
            try
            {
                File file = File.createTempFile("test_temp_dir", "");
                tempDirectory = file.getParent();
                file.delete();
            }
            catch(Throwable t)
            {
                t.printStackTrace();
            }
        }
        return tempDirectory;
    }
    
    private static void waitReportBuilderToEnd() throws InterruptedException
    {
        while(GxIsAlive())
        {
            Thread.sleep(100);
        }
    }
    
    private static void cleanupReportBuilder()
    {
        //GxShutdown();
    }
    
    private void addTempFile(String filePath)
    {
        this.log("Appending file to delete: " + filePath);
        this.tempFiles.add(filePath);
    }
    
    private void deleteTempFiles()
    {
        for (String filePath : this.tempFiles)
        {
            this.log("Deleting: " + filePath);
            if (!new File(filePath).delete())
            {
                new File(filePath).deleteOnExit();
            }
        }
        this.tempFiles.clear();
    }
    
    private String getBaseUrl()
    {
        return baseUrl;
    }
    
    private void getFileFromServer(String fileName)
    {
        this.getFileFromServer(fileName, true);
    }

    
    private void getFileFromServer(String fileName, boolean deleteOnExit)
    {
        try
        {
            this.log("Getting " + fileName + " from server");
            String fileUrl = fileName;
            if (fileName.indexOf(":") < 0)
            {
                fileName = fileName.replace(File.separatorChar, '/');
                
                fileUrl = this.getBaseUrl() + (fileName.startsWith("/")?fileName:("/" + fileName));
            }
            this.log("File url = " + fileUrl);
            URL urlBCU = new URL(fileUrl);
            
            try {
                final String filePath = getTempDirectory() + File.separator + this.fileNameFromUrl(fileUrl);
                if (deleteOnExit)
                {
                    this.addTempFile(filePath);
                }
                Path targetPath = new File( filePath).toPath();
                Files.copy(urlBCU.openStream(), targetPath, StandardCopyOption.REPLACE_EXISTING);
            } 
            catch( Exception e)
            {
                this.log("Error getting " + fileName + " from server", e);
            }
        }
        catch (Throwable t)
        {
            this.log("Error getting " + fileName + " from server", t);
            t.printStackTrace();
        }
    }
    
    private String fileNameFromUrl(String url)
    {
        int idx = url.lastIndexOf('/');
        if (idx > 0)
        {
            url = url.substring(idx + 1);
        }
        idx = url.lastIndexOf(File.separator);
        if (idx > 0)
        {
            url = url.substring(idx + 1);
        }
        return url;
    }
    
	private void loadRbuilder()
	{
            try
            {
                AccessController.doPrivileged(new PrivilegedAction() {
                    public Object run() {
                        getReportBuilderLibraries();
                        log("Loading report builder libraries");					
                        System.load(getTempDirectory() + "\\gxdib32.dll");
                        System.load(getTempDirectory() + "\\rbuilder.dll");
                        return null;
                    }
                });
            }
            catch(Throwable t)
            {
                this.log("Error initializing applet", t);
            }		
	}
    
    private boolean startsWithBOM(File aTextFile) throws IOException {
    	boolean result = false;
    	if (aTextFile.length() < BYTE_ORDER_MARK.length) return false;
    	//open as bytes here, not characters
    	int[] firstFewBytes = new int[BYTE_ORDER_MARK.length];
    	InputStream input = null;
    	try {
            input = new FileInputStream(aTextFile);
            for (int index = 0; index < BYTE_ORDER_MARK.length; ++index) {
                firstFewBytes[index] = input.read(); //read a single byte
            }
            result = Arrays.equals(firstFewBytes, BYTE_ORDER_MARK);
    	} finally {
            input.close();
    	}
    	return result;
    }

    private void stripBomFrom(String aTextFile) throws IOException {
    	File bomFile = new File(aTextFile);
    	long initialSize = bomFile.length();
    	long truncatedSize = initialSize - BYTE_ORDER_MARK.length;
    	byte[] memory = new byte[(int)(truncatedSize)];
    	InputStream input = null;
    	try {
    		input = new BufferedInputStream(new FileInputStream(bomFile));
    		input.skip(BYTE_ORDER_MARK.length);
    		int totalBytesReadIntoMemory = 0;
    		while (totalBytesReadIntoMemory < truncatedSize) 
                {
                    int bytesRemaining = (int) truncatedSize - totalBytesReadIntoMemory;
                    int bytesRead = input.read(memory, totalBytesReadIntoMemory, bytesRemaining);
                    if (bytesRead > 0) {
                        totalBytesReadIntoMemory = totalBytesReadIntoMemory + bytesRead;
                    }
    		}
    		overwriteWithoutBOM(memory, bomFile);
    	} finally {
            input.close();
    	}
    	File after = new File(aTextFile);
    	long finalSize = after.length();
    	long changeInSize = initialSize - finalSize;
    	if (changeInSize != BYTE_ORDER_MARK.length) {
    		throw new RuntimeException(
    			"Change in file size: " + changeInSize +
    			" Expected change: " + BYTE_ORDER_MARK.length);
    	}
    }

    private void overwriteWithoutBOM(
    byte[] aBytesWithoutBOM, File aTextFile) throws IOException {
    	OutputStream output = null;
    	try {
    		output = new BufferedOutputStream(new FileOutputStream(aTextFile));
    		output.write(aBytesWithoutBOM);
    	} finally {
    		output.close();
    	}
    }

    private void getReportBuilderLibraries()
    {
        this.log("Getting report builder libraries from server");
        this.getFileFromServer("printingappletsigned.jar");
        try
        {
            this.log("Unzipping report builder libraries");
            final ZipInputStream zis = new ZipInputStream(new BufferedInputStream(new FileInputStream(getTempDirectory() + "\\printingappletsigned.jar")));
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null)
            {
                String entryName = entry.getName();
                if (entryName.endsWith("dll"))
                {
                    final String filePath = getTempDirectory() + "\\" + entry.getName();
                    this.addTempFile(filePath);
                    AccessController.doPrivileged(new PrivilegedAction() {
                        int count = -1;
                        byte[] bytes = new byte[BUFFER];
                        
                        public Object run() {
                            try
                            {
                                BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(filePath), BUFFER);
                                while ((count = zis.read(bytes, 0, BUFFER)) > 0)
                                {
                                    bos.write(bytes, 0, count);
                                }
                                bos.flush();
                                bos.close();
                            }
                            catch (Throwable t)
                            {
                                t.printStackTrace();
                            }
                            return null;
                        }
                    });
                }
                zis.closeEntry();
            }
            zis.close();
        }
        catch(Throwable t)
        {
            this.log("Error storing report builder libraries", t);
        }
    }
    
    private boolean fileExists(final String filePath)
    {
        final boolean[] ret = new boolean[1];
        ret[0] = false;
        AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                ret[0] = new File(filePath).exists();
                return null;
            }
        });
        return ret[0];
    }
    
    public boolean isReady()
    {
        return this.isReady;
    }
    
    public boolean printReport(String fileUrl)
    {
        return this.printReport(fileUrl, "", "");
    }
      
    public boolean printReport(String fileUrl, String printerRule, String iniUrl)
    {
        this.log("Printing report " + fileUrl);
        try
        {
            boolean hasIni = (iniUrl != null && !iniUrl.equals(""));
            
            if (hasIni)
            {
                if (!this.fileExists(getTempDirectory() + File.separator + this.fileNameFromUrl(iniUrl)))
                {
                    this.getFileFromServer(iniUrl, false);
                }
            }
            this.getFileFromServer(fileUrl, false);
            String tempDir = getTempDirectory();
            String fileName = tempDir + File.separator + this.fileNameFromUrl(fileUrl);
			
            if (fileName.endsWith(".txt")||fileName.endsWith(".pdf"))
            {	
                if (firstTime)
                {
                    printerToUse = initPrinterDialog();
                    firstTime = false;
                }

                boolean hasPrinterRule = (printerRule != null && !printerRule.equals(""));
                if (hasPrinterRule)
                    assignPrinter( printerRule, iniUrl);

                //list all print services (installed printers)
                PrintService[] printServices = PrintServiceLookup.lookupPrintServices(null, null);
                this.log("Number of print services: " + printServices.length);

                PrintService myService = null;
                for (PrintService printer : printServices){
                    //avoid spaces in ini's printer name
                    if (printer.getName().equals(printerToUse.trim())) {
                        myService = printer;
                        break;
                    }

                    this.log("Printer: " + printer.getName()); 
                }


                //final PrintService ps = (PrintService)printers.get(printerToUse);
                final String fileNameAux = fileName;

                //if (ps != null) {
                if(myService != null){
                    //printDoc(fileNameAux, ps);
                    printDoc(fileNameAux, myService);
                }else{
                    this.log("Cannot Print!! - printer \"" + printerToUse + "\" was not found ");
                }
            }
            else
            {
                if  (isWindows())
                {
                    if (firstTimeGXR)
                    {
                        loadRbuilder();
                        firstTimeGXR = false;
                    }
                    String iniName = hasIni?tempDir:"";
                    this.log("Calling report builder dll with (\"" + fileName + "\", \"" + iniName + "\"");
                    GxPrintDoc(fileName, iniName);
                    waitReportBuilderToEnd();
                }
            }
        }
        catch(Throwable t)
        {
            this.log("Error printing report " + fileUrl, t);
            return false;
        }
        finally {
            cleanup();
        }
        return true;
    }
	
	private String printerToUse;
	
	private boolean assignPrinter(String printerRule, String iniUrl)
	{
            if (this.fileExists(getTempDirectory() + File.separator + this.fileNameFromUrl(iniUrl)))
            {
                ParseINI ini = getINIFile(iniUrl);
		if (ini.sectionExists(printerRule))
		{
                    printerToUse = ini.getProperty(printerRule, "Printer");
                    return true;
		}
            }
            else
            {
                this.log("Missing printer definition file \"" + iniUrl + "\"");
            }
            this.log("Error: No definition for printer\"" + printerRule + "\"");
            return false;
	}

        private void updateINIFile(String printerRule, String iniUrl)
	{
            ParseINI ini = getINIFile(iniUrl);
            ini.setProperty(printerRule, "Printer", printerToUse);
            final ParseINI internalINI = ini;
            AccessController.doPrivileged(new PrivilegedAction() {
                public Object run() {
                    try
                    {
                        internalINI.save();
                    }
                    catch(java.io.IOException e)
                    {
                    }						
                    return null;
                }
            });
	}
	
	private ParseINI getINIFile(String iniUrl)
	{
            final String filePath = getTempDirectory() + File.separator + this.fileNameFromUrl(iniUrl);
            ParseINI ini = AccessController.doPrivileged
            (
                 new PrivilegedAction<ParseINI>()
                {
                    public ParseINI run() 
                    {

                        try
                        {
                            return new ParseINI(filePath);
                        }
                        catch(java.io.IOException e)
                        {
                            return new ParseINI();
                        }																				  
                    }
                }
            );
            return ini;
	}
	
	private String getDefaultPrinter()
	{
            PrintService ps = PrintServiceLookup.lookupDefaultPrintService();
            if (ps != null)
            {
                printers.put("##GXDefaultPrinter##", ps);
                return ps.getName();
            }
            else
            {
                PrintService[] ps1 = PrintServiceLookup.lookupPrintServices(null, null);
                printers.put("##GXDefaultPrinter##", ps1[0]);
                return ps1[0].getName();
            }
	}
	
	java.util.Hashtable printers = new java.util.Hashtable();
	private void loadAllPrinters()
	{			
            PrintService[] ps = PrintServiceLookup.lookupPrintServices(null, null);
            for (int p = 0; p < ps.length; p++) 
            {
                ch.addItem(ps[p].getName());
                printers.put(ps[p].getName(), ps[p]);
            }
	}
	
	/** BOM for UTF-8. */
	private final int[] BYTE_ORDER_MARK = {239, 187, 191};

	private void printDoc(String fileName, PrintService ps)
	{
		if  (isWindows())
		{		
                    FileInputStream fInput = null;
                    try
                    {
                        DocFlavor[] docFlavor = ps.getSupportedDocFlavors();
                        this.log("Supported flavors:");
                        for (int i = 0; i < docFlavor.length; i++) {
                                this.log(docFlavor[i].toString());
                        }   

                        DocFlavor flavor = null;

                        if (fileName.toLowerCase().endsWith("pdf")){

                            PDDocument document = new PDDocument();

                            try {
                                document = PDDocument.load(new File(fileName));
                                PrinterJob job = PrinterJob.getPrinterJob();
                                job.setPageable(new PDFPageable(document));
                                job.setPrintService(ps);
                                job.print();
                            }
                            catch (NullPointerException | PrinterException e)
                            {	
                                    this.log("PDFBoxPrint Exception: ", e);
                            }
                            finally{
                                if (document != null)
                                    try {
                                        document.close();
                                    }
                                    catch(IOException e){
                                        this.log("Error printing pdf report " + fileName, e); 
                                }      
                            }
                        }
                        else
                        {
                            if (fileName.toLowerCase().endsWith("txt")){
                                flavor = DocFlavor.INPUT_STREAM.TEXT_PLAIN_UTF_8;					
                                if ( !ps.isDocFlavorSupported(flavor))
                                {
                                    File bomFile = new File(fileName);
                                    if (startsWithBOM(bomFile))
                                    {
                                        this.log("Strip Byte Order Mark from UTF8 file");
                                        stripBomFrom(fileName);
                                    }
                                }
                            }


                            if (flavor == null || !ps.isDocFlavorSupported(flavor)){
                                flavor = DocFlavor.INPUT_STREAM.AUTOSENSE;
                                this.log("Flavor Text Plain UTF8 NOT supported");
                            }else
                            {
                                this.log("Flavor Text Plain UTF8 supported!");
                            }
                            fInput = new FileInputStream(fileName);				
                            DocPrintJob pj = ps.createPrintJob();
                            Doc doc = new SimpleDoc(fInput, flavor, null);
                            pj.print(doc, null);
                        }
                    }
                    catch (FileNotFoundException e1)
                    {	
                        this.log("File " + fileName + " not found: ", e1);
                    }
                    catch (PrintException e) 
                    {
                        this.log("Error printing report " + fileName, e);
                    }
                    catch (Exception e2) 
                    {
                        this.log("Error printing " + fileName, e2);
                    }
                    finally {
                        if (fInput != null)
                            try {
                                fInput.close();
                            }
                            catch(IOException e){
                                this.log("Error printing report " + fileName, e); 
                        }                            
                    }
		}
		else
		{
                    try
                    {
                        String [] cmd = new String[] { "lp", "-d", ps.getName(), fileName};
                        Process p = Runtime.getRuntime().exec(cmd);
                        p.waitFor();
                    } 
                    catch(Exception e1)
                    {
                        this.log("Error printing report " + fileName, e1); 
                    }
		}
	}
	
	boolean firstTime = true;
	boolean firstTimeGXR = true;
	Dialog m_Frame;
	Frame m_Frame1;
	CheckboxGroup cg1=new CheckboxGroup();
	Checkbox deafultPrinter;
	Checkbox otherPrinter = new Checkbox("Other printer:", cg1, false);
	SelectOtherPrinter sl = new SelectOtherPrinter(this);
	Choice ch = new Choice();
	Dialog m_Frame2;
	String selectedPrinter = "";	

         private String initPrinterDialog()
	{
            String defaultPrinter = getDefaultPrinter();
            return defaultPrinter;
	}
        
	public void changeOtherPrinterText(ItemEvent ev)
	{
            if (cg1.getSelectedCheckbox().equals(otherPrinter))
            {	
                m_Frame2.setVisible(true);			
                otherPrinter.setLabel("Other printer: " + selectedPrinter);
                printerToUse = selectedPrinter;
            }
            else
            {
                printerToUse = "##GXDefaultPrinter##";
            }
	}
    
    private void createLogFile()
    {
        try
        {
            String logFileName = this.getLogFileName();
            log = Logger.getLogger(Printer.class.getName());
            log.setLevel(Level.ALL);
            if (this.loggerFile == null)
            {
                this.loggerFile = new FileHandler(logFileName);
            }
            log.addHandler(this.loggerFile);
        }
        catch (Throwable t)
        {
            t.printStackTrace();
        }
    }
    
    private String getLogFileName()
    {
        
        return getTempDirectory() + File.separator + "log" + System.currentTimeMillis() + ".log";
    }
    
    private void log(String text)
    {
        log.log(Level.ALL, text);
    }
    
    public static String getStackTrace(Throwable aThrowable) 
    {
        Writer result = new StringWriter();
        PrintWriter printWriter = new PrintWriter(result);
        aThrowable.printStackTrace(printWriter);
        return result.toString();
    }
    
    private void log(String text, Throwable t)
    {
        log.log(Level.ALL, text, t);
        log.log(Level.ALL, getStackTrace(t));
    }
    
    public void init()
    {
        this.tempFiles = new ArrayList<String>();
        createLogFile();
    }
    
    private void cleanup()
    {
        this.deleteTempFiles();
	//if  (!firstTimeGXR)
	//{		
	//	cleanupReportBuilder();
        //}
        log.removeHandler(this.loggerFile);
    }

    
    /*@Override
    public void start()
    {
        super.start();
        this.isReady = true;
    }

    public void destroy()
    {
        super.destroy();
        this.deleteTempFiles();
	if  (!firstTimeGXR)
	{		
		cleanupReportBuilder();
	}
    }
    */
	
    class GXAction implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {	
            m_Frame.setVisible(false);
        }
    }

    class GXAction1 implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            selectedPrinter = ch.getSelectedItem();
            m_Frame2.setVisible(false);
        }
    }	

    class WPWindowListener extends WindowAdapter 
    {
        public void windowClosing(WindowEvent e) {}
    }

    class SelectOtherPrinter implements ItemListener
    {
        private Printer applet;
        public SelectOtherPrinter(Printer applet)
        {
            this.applet = applet;
        }
        public void itemStateChanged(ItemEvent ev)
        {
            applet.changeOtherPrinterText(ev);
        }
    }
}
